<?php
/*
Plugin Name: FB Pixel & CAPI Pro
Plugin URI: https://verveeco.shop
Description: Pro Pixel + Conversions API integration with Order Flow events, auto fbc sync, WooCommerce hooks, test event support, and logging.
Version: 1.0.0
Author: JOY DHAR
Author URI: https://verveeco.shop
License: GPLv2 or later
Text Domain: fb-pixel-capi-pro
*/

if (!defined('ABSPATH')) exit;

/**
 * Basic constants
 */
define('FBCAPI_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('FBCAPI_PLUGIN_URL', plugin_dir_url(__FILE__));
if(!defined('FBCAPI_API_DEFAULT')) define('FBCAPI_API_DEFAULT', 'v18.0');

/**
 * Helpers
 */
function fbcapi_get_option($key, $default=''){
    return get_option('fbcapi_' . $key, $default);
}
function fbcapi_get_token(){
    if( defined('FBCAPI_ACCESS_TOKEN') && FBCAPI_ACCESS_TOKEN ){
        return FBCAPI_ACCESS_TOKEN;
    }
    return fbcapi_get_option('access_token','');
}
function fbcapi_get_appsecret(){
    if( defined('FBCAPI_APP_SECRET') && FBCAPI_APP_SECRET ){
        return FBCAPI_APP_SECRET;
    }
    return fbcapi_get_option('app_secret','');
}
function fbcapi_get_api_version(){
    return fbcapi_get_option('api_version', FBCAPI_API_DEFAULT);
}
function fbcapi_log_error($msg){
    if(function_exists('error_log')){
        error_log('FBCAPI-PRO: ' . $msg);
    }
    update_option('fbcapi_last_error', wp_json_encode(array('time'=>time(),'msg'=>$msg)));
}
function fbcapi_get_allowed_events(){
    return array(
        'PageView','ViewContent','AddToCart','InitiateCheckout','Purchase',
        'OrderConfirmed','OrderCancelled','OrderShipping','OrderReturned','OrderDelivered','OrderCompleted'
    );
}
function fbcapi_sanitize_custom_data($cd){
    if(!is_array($cd)) return $cd;
    $out = array();
    if(isset($cd['value'])) $out['value'] = floatval($cd['value']);
    if(!empty($cd['currency'])) $out['currency'] = strtoupper(substr(preg_replace('/[^A-Z]/', '', $cd['currency']),0,3));
    if(!empty($cd['content_ids'])){
        if(is_string($cd['content_ids'])) $cd['content_ids'] = array_map('trim', explode(',', $cd['content_ids']));
        if(is_array($cd['content_ids'])) $out['content_ids'] = array_map('strval', $cd['content_ids']);
    }
    foreach(array('content_name','shipping','carrier','order_id','tracking_number','delivery_date') as $k){
        if(isset($cd[$k])) $out[$k] = is_scalar($cd[$k]) ? sanitize_text_field($cd[$k]) : $cd[$k];
    }
    return $out;
}
function fbcapi_get_appsecret_proof($access_token, $app_secret){
    if(empty($access_token) || empty($app_secret)) return '';
    return hash_hmac('sha256', $access_token, $app_secret);
}

/**
 * Admin menu and settings
 */
add_action('admin_menu', function(){
    add_menu_page('Pixel & CAPI Pro', 'Pixel & CAPI Pro', 'manage_options', 'fbcapi-pro', 'fbcapi_admin_page', '', 59);
    add_submenu_page('fbcapi-pro', 'Settings', 'Settings', 'manage_options', 'fbcapi-pro-settings', 'fbcapi_admin_page');
});
add_action('admin_init', function(){
    register_setting('fbcapi_pro_settings', 'fbcapi_pixel_id');
    register_setting('fbcapi_pro_settings', 'fbcapi_access_token');
    register_setting('fbcapi_pro_settings', 'fbcapi_app_secret');
    register_setting('fbcapi_pro_settings', 'fbcapi_api_version');
    register_setting('fbcapi_pro_settings', 'fbcapi_test_event_code');
    register_setting('fbcapi_pro_settings', 'fbcapi_default_cc');
    register_setting('fbcapi_pro_settings', 'fbcapi_enable_woo_hooks', 'intval');
});

function fbcapi_admin_page(){
    if(!current_user_can('manage_options')) return;
    if($_SERVER['REQUEST_METHOD'] === 'POST' && check_admin_referer('fbcapi_save_settings')){
        // handled by register_setting
    }
    $pixel_id = fbcapi_get_option('pixel_id','');
    $token = fbcapi_get_option('access_token','');
    $app_secret = fbcapi_get_option('app_secret','');
    $api_version = fbcapi_get_option('api_version', FBCAPI_API_DEFAULT);
    $test_event = fbcapi_get_option('test_event_code','');
    $default_cc = fbcapi_get_option('default_cc','');
    $enable_woo = fbcapi_get_option('enable_woo_hooks', 1);
    $last_error = get_option('fbcapi_last_error','');
    ?>
    <div class="wrap">
        <h1>FB Pixel & CAPI Pro — Settings</h1>
        <form method="post" action="">
            <?php wp_nonce_field('fbcapi_save_settings'); ?>
            <table class="form-table">
                <tr><th>Pixel ID</th><td><input name="fbcapi_pixel_id" value="<?php echo esc_attr($pixel_id); ?>" class="regular-text"></td></tr>
                <tr><th>Access Token</th><td><input name="fbcapi_access_token" value="<?php echo esc_attr($token); ?>" class="regular-text"><p class="description">Prefer defining <code>FBCAPI_ACCESS_TOKEN</code> in wp-config.php for security.</p></td></tr>
                <tr><th>App Secret</th><td><input name="fbcapi_app_secret" value="<?php echo esc_attr($app_secret); ?>" class="regular-text"><p class="description">Or define <code>FBCAPI_APP_SECRET</code> in wp-config.php</p></td></tr>
                <tr><th>Graph API Version</th><td><input name="fbcapi_api_version" value="<?php echo esc_attr($api_version); ?>" class="regular-text"></td></tr>
                <tr><th>Test Event Code</th><td><input name="fbcapi_test_event_code" value="<?php echo esc_attr($test_event); ?>" class="regular-text"></td></tr>
                <tr><th>Default Country Code</th><td><input name="fbcapi_default_cc" value="<?php echo esc_attr($default_cc); ?>" class="regular-text"><p class="description">E.g. 880 for Bangladesh — used for phone normalization if needed.</p></td></tr>
                <tr><th>Enable WooCommerce Hooks</th><td><input type="checkbox" name="fbcapi_enable_woo_hooks" value="1" <?php checked($enable_woo,1); ?>></td></tr>
            </table>
            <?php submit_button('Save Settings'); ?>
        </form>

        <h2>Logs</h2>
        <pre style="max-height:200px; overflow:auto; background:#fff; padding:10px;"><?php echo esc_html($last_error); ?></pre>

        <h2>Quick Test</h2>
        <p>Use browser console to run: <code>fbcapiSendEvent({event_name:'Purchase', custom_data:{value:1.23, currency:'USD'}, fbc:getFbClickId()})</code></p>
    </div>
    <?php
}

/**
 * Enqueue frontend JS
 */
add_action('wp_enqueue_scripts', function(){
    wp_register_script('fbcapi-pro-js', FBCAPI_PLUGIN_URL . 'assets/js/fbcapi.js', array(), '1.0', true);
    wp_localize_script('fbcapi-pro-js', 'fbcapiRest', array(
        'url' => esc_url_raw(rest_url('fbcapi/v1/event')),
        'nonce' => wp_create_nonce('wp_rest')
    ));
    wp_enqueue_script('fbcapi-pro-js');
});

/**
 * REST endpoint
 */
add_action('rest_api_init', function(){
    register_rest_route('fbcapi/v1','/event', array(
        'methods' => 'POST',
        'callback' => 'fbcapi_rest_receive_event',
        'permission_callback' => function($request){
            $nonce = $request->get_header('x_wp_nonce') ?: $request->get_header('X-WP-Nonce');
            if($nonce && wp_verify_nonce($nonce,'wp_rest')) return true;
            return is_user_logged_in();
        }
    ));
});

function fbcapi_rest_receive_event($request){
    $body = $request->get_json_params();
    if(!$body || !isset($body['event_name'])) return new WP_Error('invalid', 'Missing event_name', array('status'=>400));
    $event_name = sanitize_text_field($body['event_name']);
    $allowed = fbcapi_get_allowed_events();
    if(!in_array($event_name, $allowed, true)){
        return new WP_Error('invalid_event','Event not allowed', array('status'=>403));
    }
    $custom = isset($body['custom_data']) && is_array($body['custom_data']) ? $body['custom_data'] : array();
    if(isset($body['fbc']) && $body['fbc']){
        $custom['fbc'] = sanitize_text_field($body['fbc']);
    }
    $res = fbcapi_send_event($event_name, array('custom_data'=>$custom, 'event_source_url'=>isset($body['event_source_url']) ? esc_url_raw($body['event_source_url']) : ''));
    if(is_wp_error($res)) return $res;
    return array('success'=>true, 'response'=>$res);
}

/**
 * send event to Graph API
 */
function fbcapi_send_event($event_name, $args = array()){
    $pixel_id = fbcapi_get_option('pixel_id','');
    if(defined('FBCAPI_PIXEL_ID') && FBCAPI_PIXEL_ID) $pixel_id = FBCAPI_PIXEL_ID;
    if(empty($pixel_id)) return new WP_Error('nopixel','No Pixel ID configured', array('status'=>500));
    $token = fbcapi_get_token();
    if(empty($token)) return new WP_Error('notoken','No access token configured', array('status'=>500));
    $api_version = fbcapi_get_api_version();
    $url = "https://graph.facebook.com/{$api_version}/{$pixel_id}/events?access_token=" . urlencode($token);

    $app_secret = fbcapi_get_appsecret();
    $proof = fbcapi_get_appsecret_proof($token, $app_secret);
    if(!empty($proof)) $url .= '&appsecret_proof=' . urlencode($proof);
    $test_code = fbcapi_get_option('test_event_code','');
    if(!empty($test_code)) $url .= '&test_event_code=' . urlencode($test_code);

    $event = array(
        'event_name' => $event_name,
        'event_time' => time(),
        'event_source_url' => isset($args['event_source_url']) ? $args['event_source_url'] : '',
    );
    $custom = isset($args['custom_data']) ? $args['custom_data'] : array();
    $event['custom_data'] = fbcapi_sanitize_custom_data($custom);

    $user_data = isset($args['user_data']) && is_array($args['user_data']) ? $args['user_data'] : array();
    if(isset($user_data['email'])) $user_data['email'] = strtolower(trim($user_data['email']));
    if(isset($user_data['phone'])) $user_data['phone'] = preg_replace('/\D+/', '', $user_data['phone']);
    if(!empty($user_data)) $event['user_data'] = $user_data;

    if(isset($event['custom_data']['fbc']) && $event['custom_data']['fbc']){
        if(!isset($event['user_data'])) $event['user_data'] = array();
        $event['user_data']['fbc'] = $event['custom_data']['fbc'];
    }

    $payload = array('data'=> array($event));
    $body = wp_json_encode($payload);

    $resp = wp_remote_post($url, array(
        'headers' => array('Content-Type'=>'application/json'),
        'body' => $body,
        'timeout' => 10,
    ));
    if(is_wp_error($resp) || wp_remote_retrieve_response_code($resp) >= 400){
        fbcapi_log_error(print_r($resp, true));
        return new WP_Error('api_error','Graph API error', array('status'=>500, 'details'=>$resp));
    }
    return $resp;
}

/**
 * WooCommerce hooks (if enabled)
 */
add_action('init', function(){
    $enable = fbcapi_get_option('enable_woo_hooks', 1);
    if($enable && class_exists('WooCommerce')){
        add_action('woocommerce_order_status_processing', 'fbcapi_wc_order_confirmed', 10, 1);
        add_action('woocommerce_order_status_completed', 'fbcapi_wc_order_completed', 10, 1);
        add_action('woocommerce_order_status_cancelled', 'fbcapi_wc_order_cancelled', 10, 1);
        add_action('woocommerce_order_status_refunded', 'fbcapi_wc_order_returned', 10, 1);
        add_action('woocommerce_order_status_shipped', 'fbcapi_wc_order_shipping', 10, 1);
        add_action('woocommerce_order_status_delivered', 'fbcapi_wc_order_delivered', 10, 1);
    }
});
function fbcapi_wc_send_for_order($order_id, $event_name){
    if(!function_exists('wc_get_order')) return;
    $order = wc_get_order($order_id);
    if(!$order) return;
    $data = array(
        'order_id' => strval($order_id),
        'value' => floatval($order->get_total()),
        'currency' => $order->get_currency(),
        'email' => $order->get_billing_email(),
    );
    $tracking = get_post_meta($order_id, '_tracking_number', true);
    if($tracking) $data['tracking_number'] = $tracking;
    return fbcapi_send_event($event_name, array('custom_data'=>$data, 'event_source_url'=>get_permalink($order_id)));
}
function fbcapi_wc_order_confirmed($order_id){ fbcapi_wc_send_for_order($order_id, 'OrderConfirmed'); }
function fbcapi_wc_order_completed($order_id){ fbcapi_wc_send_for_order($order_id, 'OrderCompleted'); }
function fbcapi_wc_order_cancelled($order_id){ fbcapi_wc_send_for_order($order_id, 'OrderCancelled'); }
function fbcapi_wc_order_returned($order_id){ fbcapi_wc_send_for_order($order_id, 'OrderReturned'); }
function fbcapi_wc_order_shipping($order_id){ fbcapi_wc_send_for_order($order_id, 'OrderShipping'); }
function fbcapi_wc_order_delivered($order_id){ fbcapi_wc_send_for_order($order_id, 'OrderDelivered'); }

/**
 * Activation / deactivation
 */
register_activation_hook(__FILE__, function(){
    if(get_option('fbcapi_api_version') === false) update_option('fbcapi_api_version', FBCAPI_API_DEFAULT);
    if(get_option('fbcapi_enable_woo_hooks') === false) update_option('fbcapi_enable_woo_hooks', 1);
});
register_deactivation_hook(__FILE__, function(){
    // no destructive actions
});
